import { AxiosInstance, AxiosRequestHeaders, InternalAxiosRequestConfig, Method } from "axios";
import { CredentialsProvider } from ".";
export interface InterceptorOptions {
    /**
     * Target service. Will use default aws4 behavior if not given.
     */
    service?: string;
    /**
     * AWS region name. Will use default aws4 behavior if not given.
     */
    region?: string;
    /**
     * Whether to sign query instead of adding Authorization header. Default to false.
     */
    signQuery?: boolean;
    /**
     * Whether to add a X-Amz-Content-Sha256 header.
     */
    addContentSha?: boolean;
    /**
     * ARN of the IAM Role to be assumed to get the credentials from.
     * The credentials will be cached and automatically refreshed as needed.
     * Will not be used if credentials are provided.
     */
    assumeRoleArn?: string;
    /**
     * Number of seconds before the assumed Role expiration
     * to invalidate the cache.
     * Used only if assumeRoleArn is provided.
     */
    assumedRoleExpirationMarginSec?: number;
    /**
     * An identifier for the assumed role session.
     * Use the role session name to uniquely identify a session when the same role is
     * assumed by different principals or for different reasons.
     * In cross-account scenarios, the role session name is visible to,
     * and can be logged by the account that owns the role.
     */
    assumeRoleSessionName?: string;
}
export interface SigningOptions {
    host?: string;
    headers?: AxiosRequestHeaders;
    path?: string;
    body?: unknown;
    region?: string;
    service?: string;
    signQuery?: boolean;
    method?: string;
}
export interface Credentials {
    accessKeyId: string;
    secretAccessKey: string;
    sessionToken?: string;
}
export type InternalAxiosHeaders = Record<Method | "common", Record<string, string>>;
/**
 * Create an interceptor to add to the Axios request chain. This interceptor
 * will sign requests with the AWSv4 signature.
 *
 * @example
 * axios.interceptors.request.use(
 *     aws4Interceptor({ region: "eu-west-2", service: "execute-api" })
 * );
 *
 * @param options The options to be used when signing a request
 * @param credentials Credentials to be used to sign the request
 */
export declare const aws4Interceptor: <D = any>({ instance, credentials, options, }: {
    instance?: AxiosInstance;
    options?: InterceptorOptions;
    credentials?: Credentials | CredentialsProvider;
}) => ((config: InternalAxiosRequestConfig<D>) => Promise<InternalAxiosRequestConfig<D>>);
