"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findPartition = findPartition;
const fsLabel_1 = require("./fsLabel");
/**
 * @summary Find the partition on the provided disk with a name that matches one
 * of the provided names.
 *
 * Matches on partition name for GPT partitions or on filesystem label for MBR
 * partitions.
 *
 * @returns A FindPartitionResult with the found index and name/label; otherwise
 * undefined
 */
async function findPartition(fileDisk, partitionInfo, names) {
    if (partitionInfo.type === 'gpt') {
        const partition = partitionInfo.partitions.find((gptPartInfo) => names.includes(gptPartInfo.name));
        if (partition && typeof partition.index === 'number') {
            return {
                index: partition.index,
                name: partition.name,
            };
        }
    }
    else {
        // MBR
        for (const partition of partitionInfo.partitions) {
            try {
                const label = await (0, fsLabel_1.getFsLabel)(fileDisk, partition);
                if (names.includes(label) && typeof partition.index === 'number') {
                    return {
                        index: partition.index,
                        name: label,
                    };
                }
            }
            catch (e) {
                // LabelNotFound is expected and not fatal.
                if (!(e instanceof fsLabel_1.LabelNotFound)) {
                    throw e;
                }
            }
        }
    }
}
//# sourceMappingURL=utils.js.map