"use strict";
/*
 * Copyright 2016 balena.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.hasDriveImageCompatibilityStatus = exports.getListDriveImageCompatibilityStatuses = exports.getDriveImageCompatibilityStatuses = exports.statuses = exports.COMPATIBILITY_STATUS_TYPES = exports.isDriveSizeLarge = exports.LARGE_DRIVE_SIZE = exports.isDriveSizeRecommended = exports.isDriveValid = exports.isDriveLargeEnough = exports.isSourceDrive = exports.isSystemDrive = void 0;
const lodash_1 = require("lodash");
const pathIsInside = __importStar(require("path-is-inside"));
const messages = __importStar(require("./messages"));
/**
 * @summary The default unknown size for things such as images and drives
 */
const UNKNOWN_SIZE = 0;
/**
 * @summary Check if a drive is a system drive
 */
function isSystemDrive(drive) {
    return Boolean(drive.isSystem);
}
exports.isSystemDrive = isSystemDrive;
function sourceIsInsideDrive(source, drive) {
    for (const mountpoint of drive.mountpoints || []) {
        if (pathIsInside(source, mountpoint.path)) {
            return true;
        }
    }
    return false;
}
/**
 * @summary Check if a drive is source drive
 *
 * @description
 * In the context of Etcher, a source drive is a drive
 * containing the image.
 */
function isSourceDrive(drive, selection) {
    if (selection) {
        if (selection.drive) {
            return selection.drive.device === drive.device;
        }
        if (selection.path) {
            return sourceIsInsideDrive(selection.path, drive);
        }
    }
    return false;
}
exports.isSourceDrive = isSourceDrive;
/**
 * @summary Check if a drive is large enough for an image
 */
function isDriveLargeEnough(drive, image) {
    const driveSize = drive.size || UNKNOWN_SIZE;
    if (image === undefined) {
        return true;
    }
    if (image.isSizeEstimated) {
        // If the drive size is smaller than the original image size, and
        // the final image size is just an estimation, then we stop right
        // here, based on the assumption that the final size will never
        // be less than the original size.
        if (driveSize < (image.compressedSize || UNKNOWN_SIZE)) {
            return false;
        }
        // If the final image size is just an estimation then consider it
        // large enough. In the worst case, the user gets an error saying
        // the drive has ran out of space, instead of prohibiting the flash
        // at all, when the estimation may be wrong.
        return true;
    }
    return driveSize >= (image.size || UNKNOWN_SIZE);
}
exports.isDriveLargeEnough = isDriveLargeEnough;
/**
 * @summary Check if a drive is valid, i.e. large enough for an image
 */
function isDriveValid(drive, image, write = true) {
    return (!write ||
        (!drive.disabled &&
            isDriveLargeEnough(drive, image) &&
            !isSourceDrive(drive, image)));
}
exports.isDriveValid = isDriveValid;
/**
 * @summary Check if a drive meets the recommended drive size suggestion
 *
 * @description
 * If the image doesn't have a recommended size, this function returns true.
 */
function isDriveSizeRecommended(drive, image) {
    const driveSize = drive.size || UNKNOWN_SIZE;
    return driveSize >= (image?.recommendedDriveSize || UNKNOWN_SIZE);
}
exports.isDriveSizeRecommended = isDriveSizeRecommended;
/**
 * @summary 128GB
 */
exports.LARGE_DRIVE_SIZE = 128e9;
/**
 * @summary Check whether a drive's size is 'large'
 */
function isDriveSizeLarge(drive) {
    const driveSize = drive.size || UNKNOWN_SIZE;
    return driveSize > exports.LARGE_DRIVE_SIZE;
}
exports.isDriveSizeLarge = isDriveSizeLarge;
/**
 * @summary Drive/image compatibility status types.
 *
 * @description
 * Status types classifying what kind of message it is, i.e. error, warning.
 */
exports.COMPATIBILITY_STATUS_TYPES = {
    WARNING: 1,
    ERROR: 2,
};
exports.statuses = {
    locked: {
        type: exports.COMPATIBILITY_STATUS_TYPES.ERROR,
        message: messages.compatibility.locked(),
    },
    system: {
        type: exports.COMPATIBILITY_STATUS_TYPES.WARNING,
        message: messages.compatibility.system(),
    },
    containsImage: {
        type: exports.COMPATIBILITY_STATUS_TYPES.ERROR,
        message: messages.compatibility.containsImage(),
    },
    large: {
        type: exports.COMPATIBILITY_STATUS_TYPES.WARNING,
        message: messages.compatibility.largeDrive(),
    },
    small: {
        type: exports.COMPATIBILITY_STATUS_TYPES.ERROR,
        message: messages.compatibility.tooSmall(),
    },
    sizeNotRecommended: {
        type: exports.COMPATIBILITY_STATUS_TYPES.WARNING,
        message: messages.compatibility.sizeNotRecommended(),
    },
};
/**
 * @summary Get drive/image compatibility in an object
 *
 * @description
 * Given an image and a drive, return their compatibility status object
 * containing the status type (ERROR, WARNING), and accompanying
 * status message.
 *
 * @returns {Object[]} list of compatibility status objects
 */
function getDriveImageCompatibilityStatuses(drive, image, write) {
    const statusList = [];
    // Mind the order of the if-statements if you modify.
    if (drive.isReadOnly && write) {
        statusList.push({
            type: exports.COMPATIBILITY_STATUS_TYPES.ERROR,
            message: messages.compatibility.locked(),
        });
    }
    if (!(0, lodash_1.isNil)(drive) &&
        !(0, lodash_1.isNil)(drive.size) &&
        !isDriveLargeEnough(drive, image)) {
        statusList.push(exports.statuses.small);
    }
    else {
        // Avoid showing "large drive" with "system drive" status
        if (isSystemDrive(drive)) {
            statusList.push(exports.statuses.system);
        }
        else if (isDriveSizeLarge(drive)) {
            statusList.push(exports.statuses.large);
        }
        if (isSourceDrive(drive, image)) {
            statusList.push(exports.statuses.containsImage);
        }
        if (image !== undefined &&
            !(0, lodash_1.isNil)(drive) &&
            !isDriveSizeRecommended(drive, image)) {
            statusList.push(exports.statuses.sizeNotRecommended);
        }
    }
    return statusList;
}
exports.getDriveImageCompatibilityStatuses = getDriveImageCompatibilityStatuses;
/**
 * @summary Get drive/image compatibility status for many drives
 *
 * @description
 * Given an image and a list of drives, return all compatibility status objects,
 * containing the status type (ERROR, WARNING), and accompanying status message.
 */
function getListDriveImageCompatibilityStatuses(drives, image, write) {
    return drives.flatMap((drive) => {
        return getDriveImageCompatibilityStatuses(drive, image, write);
    });
}
exports.getListDriveImageCompatibilityStatuses = getListDriveImageCompatibilityStatuses;
/**
 * @summary Does the drive/image pair have at least one compatibility status?
 *
 * @description
 * Given an image and a drive, return whether they have a connected compatibility status object.
 */
function hasDriveImageCompatibilityStatus(drive, image, write) {
    return Boolean(getDriveImageCompatibilityStatuses(drive, image, write).length);
}
exports.hasDriveImageCompatibilityStatus = hasDriveImageCompatibilityStatus;
