"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Block = void 0;
const constants_1 = require("./constants");
const uint64_1 = require("./uint64");
/**
 * Mish Blkx Block Descriptor
 */
class Block {
    constructor() {
        /** Entry / compression type */
        this.type = 0x00000000;
        /** Entry type name */
        this.description = 'UNKNOWN';
        /** Comment ('+beg'|'+end' if type == COMMENT) */
        this.comment = '';
        /** Start sector of this chunk */
        this.sectorNumber = 0x0000000000000000;
        /** Number of sectors in this chunk */
        this.sectorCount = 0x0000000000000000;
        /** Start of chunk in data fork */
        this.compressedOffset = 0x0000000000000000;
        /** Chunk's bytelength in data fork */
        this.compressedLength = 0x0000000000000000;
    }
    /**
     * Get a human readable block type
     */
    static getDescription(type) {
        switch (type) {
            case constants_1.BLOCK.ZEROFILL:
                return 'ZEROFILL';
                break;
            case constants_1.BLOCK.RAW:
                return 'UDRW (raw)';
                break;
            case constants_1.BLOCK.FREE:
                return 'FREE (unallocated)';
                break;
            case constants_1.BLOCK.UDCO:
                return 'UDCO (adc-compressed)';
                break;
            case constants_1.BLOCK.UDZO:
                return 'UDZO (zlib-compressed)';
                break;
            case constants_1.BLOCK.UDBZ:
                return 'UDBZ (bzip2-compressed)';
                break;
            case constants_1.BLOCK.COMMENT:
                return 'COMMENT';
                break;
            case constants_1.BLOCK.TERMINATOR:
                return 'TERMINATOR';
                break;
            default:
                return 'UNKNOWN';
                break;
        }
    }
    /**
     * Parse Mish Block data from a buffer
     */
    static parse(buffer, offset = 0) {
        return new Block().parse(buffer, offset);
    }
    /**
     * Parse Mish Block data from a buffer
     */
    parse(buffer, offset = 0) {
        this.type = buffer.readUInt32BE(offset + 0);
        this.description = Block.getDescription(this.type);
        this.comment = buffer
            .toString('ascii', offset + 4, offset + 8)
            .replace(/\u0000/g, '');
        this.sectorNumber = (0, uint64_1.readBE)(buffer, offset + 8);
        this.sectorCount = (0, uint64_1.readBE)(buffer, offset + 16);
        this.compressedOffset = (0, uint64_1.readBE)(buffer, offset + 24);
        this.compressedLength = (0, uint64_1.readBE)(buffer, offset + 32);
        return this;
    }
    /**
     * Write Mish Block data to a buffer
     */
    write(buffer, offset = 0) {
        buffer.writeUInt32BE(this.type, offset + 0);
        buffer.write(this.comment, offset + 4, 4, 'ascii');
        (0, uint64_1.writeBE)(buffer, this.sectorNumber, offset + 8);
        (0, uint64_1.writeBE)(buffer, this.sectorCount, offset + 16);
        (0, uint64_1.writeBE)(buffer, this.compressedOffset, offset + 24);
        (0, uint64_1.writeBE)(buffer, this.compressedLength, offset + 32);
        return buffer;
    }
}
exports.Block = Block;
/**
 * Size of the Mish Blkx Block Descriptor in bytes
 */
Block.SIZE = 40;
//# sourceMappingURL=block.js.map