"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Footer = void 0;
const checksum_1 = require("./checksum");
const uint64_1 = require("./uint64");
/**
 * UDIF Footer (aka Koly Block)
 */
class Footer {
    constructor() {
        /** Signature */
        this.signature = Footer.SIGNATURE;
        /** Version number */
        this.version = 0;
        /** Size of this structure in bytes */
        this.headerSize = 0;
        /** Flags */
        this.flags = 0;
        /** Running data fork offset (?) */
        this.runningDataForkOffset = 0;
        /** Data fork offset */
        this.dataForkOffset = 0;
        /** Data fork length in bytes */
        this.dataForkLength = 0;
        /** Resource fork offset */
        this.resourceForkOffset = 0;
        /** Resource fork length in bytes */
        this.resourceForkLength = 0;
        /** Segment number */
        this.segmentNumber = 0;
        /** Segment count */
        this.segmentCount = 0;
        /** Segment ID */
        this.segmentId = Buffer.alloc(16);
        /** Data checksum */
        this.dataChecksum = new checksum_1.Checksum();
        /** XML data offset */
        this.xmlOffset = 0;
        /** XML data length in bytes */
        this.xmlLength = 0;
        /** Reserved 1 */
        this.reserved1 = Buffer.alloc(120);
        /** Checksum */
        this.checksum = new checksum_1.Checksum();
        /** Image variant */
        this.imageVariant = 0;
        /** Sector count */
        this.sectorCount = 0;
        /** Reseved 2 */
        this.reserved2 = 0;
        /** Reseved 3 */
        this.reserved3 = 0;
        /** Reseved 4 */
        this.reserved4 = 0;
    }
    /** Parse a Koly block from a buffer */
    static parse(buffer, offset = 0) {
        return new Footer().parse(buffer, offset);
    }
    /** Parse a Koly block from a buffer */
    parse(buffer, offset = 0) {
        this.signature = buffer.readUInt32BE(offset + 0);
        if (this.signature !== Footer.SIGNATURE) {
            const expected = Footer.SIGNATURE.toString(16);
            const actual = this.signature.toString(16);
            throw new Error(`Invalid footer signature: Expected 0x${expected}, saw 0x${actual}`);
        }
        this.version = buffer.readUInt32BE(offset + 4);
        this.headerSize = buffer.readUInt32BE(offset + 8);
        this.flags = buffer.readUInt32BE(offset + 12);
        this.runningDataForkOffset = (0, uint64_1.readBE)(buffer, offset + 16);
        this.dataForkOffset = (0, uint64_1.readBE)(buffer, offset + 24);
        this.dataForkLength = (0, uint64_1.readBE)(buffer, offset + 32);
        this.resourceForkOffset = (0, uint64_1.readBE)(buffer, offset + 40);
        this.resourceForkLength = (0, uint64_1.readBE)(buffer, offset + 48);
        this.segmentNumber = buffer.readUInt32BE(offset + 56);
        this.segmentCount = buffer.readUInt32BE(offset + 60);
        buffer.copy(this.segmentId, 0, offset + 64, offset + 64 + 16);
        this.dataChecksum.parse(buffer, offset + 80);
        this.xmlOffset = (0, uint64_1.readBE)(buffer, offset + 216);
        this.xmlLength = (0, uint64_1.readBE)(buffer, offset + 224);
        buffer.copy(this.reserved1, 0, offset + 232, offset + 232 + 120);
        this.checksum.parse(buffer, offset + 352);
        this.imageVariant = buffer.readUInt32BE(offset + 488);
        this.sectorCount = (0, uint64_1.readBE)(buffer, offset + 492);
        this.reserved2 = buffer.readUInt32BE(offset + 500);
        this.reserved3 = buffer.readUInt32BE(offset + 504);
        this.reserved4 = buffer.readUInt32BE(offset + 508);
        return this;
    }
    /**
     * Write a Koly block to a buffer
     */
    write(buffer, offset = 0) {
        buffer = buffer || Buffer.alloc(Footer.SIZE + offset);
        buffer.writeUInt32BE(this.signature, offset + 0);
        buffer.writeUInt32BE(this.version, offset + 4);
        buffer.writeUInt32BE(this.headerSize, offset + 8);
        buffer.writeUInt32BE(this.flags, offset + 12);
        (0, uint64_1.writeBE)(buffer, this.runningDataForkOffset, offset + 16);
        (0, uint64_1.writeBE)(buffer, this.dataForkOffset, offset + 24);
        (0, uint64_1.writeBE)(buffer, this.dataForkLength, offset + 32);
        (0, uint64_1.writeBE)(buffer, this.resourceForkOffset, offset + 40);
        (0, uint64_1.writeBE)(buffer, this.resourceForkLength, offset + 48);
        buffer.writeUInt32BE(this.segmentNumber, offset + 56);
        buffer.writeUInt32BE(this.segmentCount, offset + 60);
        this.segmentId.copy(buffer, offset + 64);
        this.dataChecksum.write(buffer, offset + 80);
        (0, uint64_1.writeBE)(buffer, this.xmlOffset, offset + 216);
        (0, uint64_1.writeBE)(buffer, this.xmlLength, offset + 224);
        this.reserved1.copy(buffer, offset + 232);
        this.checksum.write(buffer, offset + 352);
        buffer.writeUInt32BE(this.imageVariant, offset + 488);
        (0, uint64_1.writeBE)(buffer, this.sectorCount, offset + 492);
        buffer.writeUInt32BE(this.reserved2, offset + 500);
        buffer.writeUInt32BE(this.reserved3, offset + 504);
        buffer.writeUInt32BE(this.reserved4, offset + 508);
        return buffer;
    }
}
exports.Footer = Footer;
/** Koly block structure size in bytes */
Footer.SIZE = 512;
/** Magic 32-bit value; 'koly' in ASCII */
Footer.SIGNATURE = 0x6b6f6c79;
/** UDIF Version (latest known == 4) */
Footer.VERSION = 0x00000004;
//# sourceMappingURL=footer.js.map